<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'tel',
        'role',
        'photo',
        'sex',
        'position',
        'last_connection',
        'active',
        'specialty_id',
        'profil',
        'is_doctor',
        'company_id',
        'custom_id'
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    public function interventions(){
        return $this->hasMany(Intervention::class)->orderBy('id', 'desc');
    }

    public function consultations(){
        return $this->hasMany(Consultation::class)->orderBy('id', 'desc');
    }

    public function rdvs(){
        return $this->hasMany(Rdv::class)->orderBy('id', 'desc');
    }

    public function isAdmin(){
        return ($this->profil == 'admin' OR $this->profil == 'super admin');
    }

    public function isSecretary(){
        return ($this->profil == 'secretaire');
    }

    public function isDoctor(){
        return ($this->is_doctor);
    }

    public function isTraitant(){
        return ($this->profil == 'traitant');
    }

    public function specialty(){
        return $this->belongsTo(Specialty::class);
    }

    public function firstname(){
        return explode(' ', $this->name)[0];
    }

    public function lastname(){
        $lastname = '';
        $i = 0;
        foreach (explode(' ', $this->name) as $word){
            if($i != 0) $lastname .= $word;
            $i++;
        }
        return $lastname;
    }

    public function company(){
        return $this->belongsTo(Company::class);
    }

    public function subBasic(){
        return in_array(auth()->user()->company->subscription->name, ['basic', 'standard', 'gold']);
    }

    public function subStandard(){
        return in_array(auth()->user()->company->subscription->name, ['standard', 'gold']);
    }

    public function subGold(){
        return auth()->user()->company->subscription->name == 'gold';
    }

    public function isSA(){
        return auth()->user()->profil == 'super admin';
    }

    public static function findBySlug($slug)
    {
        return static::where('custom_id', $slug)->first();
    }

    public function hasRelatedRecords(){
        if($this->interventions()->exists())
            return true;
        if($this->consultations()->exists())
            return true;
        if($this->rdvs()->exists())
            return true;
        return false;
    }

    public function isSubscriber(){
        return ($this->company->subscriber_id == $this->id);
    }

    public function proName(){
        return $this->is_doctor ? 'Dr. '.$this->name : $this->name;
    }

}
