<?php

namespace App\Http\Controllers\Admin;
use App\Http\Controllers\controller;

use App\Mail\NewAccountMail;
use App\Models\Company;
use App\Models\Specialty;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $datas = User::where('company_id', auth()->user()->company_id)->orderBy('role')->get();
        $dataRoles = [];
        foreach ($datas as $data){
            $dataRoles[$data->role][] = $data;
        }
        return view('user.list', compact('datas', 'dataRoles'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        if(auth()->user()->company->isUsersMax())
            return $this->index();
        $specialties = Specialty::orderBy('name')->get();
        return view('user.create', compact('specialties'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $data = $request->all();
        if($request->file('photo')) {
            $data['photo'] = $this->storeFile($request->file('photo'));
        }
        $data['name'] = ucfirst(strtolower($data['firstname'])).' '.strtoupper($data['lastname']);
        $data['password'] = Hash::make(strtolower($data['firstname']));
        $data['active'] = (isset($data['active'])) ? true : false;
        $data['is_doctor'] = (isset($data['is_doctor'])) ? true : false;
        $data['company_id'] = auth()->user()->company_id;
        $data['custom_id'] = $this->generateCustomID();
        $user = User::create($data);

        // envoyer mail de creation de compte au nouvel utilisateur
        $mailData = [
            'new_user' => $user,
            'company' => $user->company,
        ];
        Mail::to($user)->send(new NewAccountMail($mailData));

        return $this->index();
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\Response
     */
    public function show(User $user)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\Response
     */
    public function edit($user)
    {
        $data = User::findBySlug($user);
        $specialties = Specialty::orderBy('name')->get();
        return view('user.edit', compact('specialties', 'data'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $user)
    {
        $user = User::findBySlug($user);
        $data = $request->all();
        if($request->file('photo')) {
            $data['photo'] = $this->storeFile($request->file('photo'));
        }
        $data['name'] = ucfirst(strtolower($data['firstname'])).' '.strtoupper($data['lastname']);
        if(isset($data['password']) AND $data['password'] != null) {
            $data['password'] = Hash::make($data['password']);
        }else{
            $data['password'] = $user->password;
        }

        $data['active'] = isset($data['active']) ? true : false;
        $data['is_doctor'] = isset($data['is_doctor']) ? true : false;
        $user->update($data);
        return auth()->user()->isAdmin() ? $this->index() : $this->edit($user);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\Response
     */
    public function destroy($user)
    {
        $user = User::findBySlug($user);
        $user->delete();
        return $this->index();
    }

    private function storeFile($file){
        $name = $file->hashName();
        $file->move(storage_path('app/public/uploads/users'), $name);
        return $name;
    }

    public function checkPassword(User $user, Request $request)
    {
        $password = $request->get('password');
        if(password_verify($password, $user->password))
            return true;
        return false;
    }
}
